import React, { useEffect, useState } from 'react';
import {
    Image,
    Text,
    TouchableOpacity,
    View,
} from 'react-native';
import { styles } from './HomeScreen.styles';

const fallbackImg = 'https://secure.espncdn.com/combiner/i?img=/i/headshots/nophoto.png';

const getLogoUrl = (rotation: string, date: string) =>
  `https://www.sportsbettingonline.ag/engine/sbo/images/live_team_logos/${date}/${rotation}.jpg`;

const getPitcherImg = (id: string) =>
  `https://a.espncdn.com/combiner/i?img=/i/headshots/mlb/players/full/${id}.png`;

const checkImageExists = async (url: string): Promise<string> => {
  try {
    const res = await fetch(url, { method: 'HEAD' });
    return res.ok ? url : fallbackImg;
  } catch {
    return fallbackImg;
  }
};

export default function GameCard({
  item,
  formattedDateForURL,
  expandedId,
  setExpandedId,
  details,
  fetchDetails,
}) {
  const isExpanded = expandedId === item.id;
  const gameDetails = details[item.id];

  const [awayImg, setAwayImg] = useState(fallbackImg);
  const [homeImg, setHomeImg] = useState(fallbackImg);

  useEffect(() => {
    if (isExpanded && gameDetails) {
      const loadImages = async () => {
        const away = await checkImageExists(getPitcherImg(gameDetails.p_away_espn));
        const home = await checkImageExists(getPitcherImg(gameDetails.p_home_espn));
        setAwayImg(away);
        setHomeImg(home);
      };
      loadImages();
    }
  }, [isExpanded, gameDetails]);

  return (
    <View style={styles.card}>
      <View style={styles.row}>
        <Image source={{ uri: getLogoUrl(item.away_rotation, formattedDateForURL) }} style={styles.logo} />
        <Text style={styles.teamText}>{item.away}</Text>
        <Text style={styles.time}>{item.time}</Text>
      </View>
      <View style={styles.row}>
        <Image source={{ uri: getLogoUrl(item.home_rotation, formattedDateForURL) }} style={styles.logo} />
        <Text style={styles.teamText}>{item.home}</Text>
      </View>
      <TouchableOpacity
        style={styles.expandButton}
        onPress={() => {
          if (!gameDetails) fetchDetails(item.id);
          setExpandedId(isExpanded ? null : item.id);
        }}
      >
        <Text style={styles.expandText}>{isExpanded ? 'Hide ⬆️' : 'View more ⬇️'}</Text>
      </TouchableOpacity>

      {isExpanded && gameDetails && (
        <View style={styles.detailSection}>
          {/* Away pitcher */}
          <View style={styles.pitcherRow}>
            <Image source={{ uri: awayImg }} style={styles.pitcherImg} />
            <View>
              <Text style={styles.statText}>{gameDetails.p_away}</Text>
              <Text style={styles.statText}>ERA: {gameDetails.p_away_era_avg}</Text>
              <Text style={styles.statText}>WHIP: {gameDetails.p_away_whip_avg}</Text>
              {gameDetails.p_away_throw === 'L' && (
                <Text style={styles.leftyText}>Lefty</Text>
              )}
            </View>
          </View>

          {/* Home pitcher */}
          <View style={styles.pitcherRow}>
            <Image source={{ uri: homeImg }} style={styles.pitcherImg} />
            <View>
              <Text style={styles.statText}>{gameDetails.p_home}</Text>
              <Text style={styles.statText}>ERA: {gameDetails.p_home_era_avg}</Text>
              <Text style={styles.statText}>WHIP: {gameDetails.p_home_whip_avg}</Text>
              {gameDetails.p_home_throw === 'L' && (
                <Text style={styles.leftyText}>Lefty</Text>
              )}
            </View>
          </View>


          {/* WHEATER */}
          {gameDetails.weather && (
  <View style={styles.weatherSection}>
    <Text style={styles.weatherTitle}>🌤 Weather</Text>
    <View style={styles.weatherRow}>
      <Image
        source={{ uri: gameDetails.weather.img_url }}
        style={styles.weatherIcon}
      />
      <View>
        <Text style={styles.weatherText}>Temp: {gameDetails.weather.temp}°</Text>
        <Text style={styles.weatherText}>Humidity: {gameDetails.weather.humidity}%</Text>
        <Text style={styles.weatherText}>Wind: {gameDetails.weather.wind_speed} mph ({gameDetails.weather.wind_direction})</Text>
        <Text style={styles.weatherText}>Condition: {gameDetails.weather.condition}</Text>
      </View>
    </View>
  </View>
)}


          {/* Umpire */}
          {gameDetails.umpireDetail && (
            <View style={styles.umpireSection}>
              <Text style={styles.umpireTitle}>👨‍⚖️ Umpire</Text>
              <Text style={styles.umpireText}>Name: {gameDetails.umpireDetail.espn_name}</Text>
              <Text style={styles.umpireText}>Games: {gameDetails.umpireDetail.games}</Text>
              <Text style={styles.umpireText}>K%: {gameDetails.umpireDetail.k_pct}</Text>
              <Text style={styles.umpireText}>BB%: {gameDetails.umpireDetail.bb_pct}</Text>
            </View>
          )}
        </View>
      )}
    </View>
  );
}
