"use strict";

import * as React from 'react';
import { Dimensions, Platform, StyleSheet } from 'react-native';
// eslint-disable-next-line no-restricted-imports
import { useSafeAreaFrame } from 'react-native-safe-area-context';
import useLatestCallback from 'use-latest-callback';
import { useSyncExternalStoreWithSelector } from 'use-sync-external-store/with-selector';
import { jsx as _jsx, jsxs as _jsxs } from "react/jsx-runtime";
const FrameContext = /*#__PURE__*/React.createContext(undefined);
export function useFrameSize(selector, debounce) {
  const context = React.useContext(FrameContext);
  if (context == null) {
    throw new Error('useFrameSize must be used within a FrameSizeProvider');
  }
  const value = useSyncExternalStoreWithSelector(debounce ? context.subscribeDebounced : context.subscribe, context.getCurrent, context.getCurrent, selector);
  return value;
}
export function FrameSizeProvider({
  children
}) {
  const context = React.useContext(FrameContext);
  if (context != null) {
    // If the context is already present, don't wrap again
    return children;
  }
  return /*#__PURE__*/_jsx(FrameSizeProviderInner, {
    children: children
  });
}
function FrameSizeProviderInner({
  children
}) {
  const listeners = React.useRef(new Set());
  const {
    element,
    get
  } = useResizeListener(size => {
    listeners.current.forEach(listener => listener(size));
  });
  const getCurrent = useLatestCallback(get);
  const subscribe = useLatestCallback(listener => {
    listeners.current.add(listener);
    return () => {
      listeners.current.delete(listener);
    };
  });
  const subscribeDebounced = useLatestCallback(listener => {
    let timer;
    const debouncedListener = size => {
      clearTimeout(timer);
      timer = setTimeout(() => {
        listener(size);
      }, 100);
    };
    listeners.current.add(debouncedListener);
    return () => {
      clearTimeout(timer);
      listeners.current.delete(debouncedListener);
    };
  });
  const context = React.useMemo(() => ({
    getCurrent,
    subscribe,
    subscribeDebounced
  }), [subscribe, subscribeDebounced, getCurrent]);
  return /*#__PURE__*/_jsxs(FrameContext.Provider, {
    value: context,
    children: [element, children]
  });
}
const useResizeListener = Platform.OS === 'web' ? useResizeListenerWeb : useResizeListenerNative;
function useResizeListenerNative(onChange) {
  const frame = useSafeAreaFrame();
  React.useLayoutEffect(() => {
    onChange(frame);
  }, [frame, onChange]);
  return {
    element: null,
    get: () => frame
  };
}
const {
  width = 0,
  height = 0
} = Dimensions.get('window');

// FIXME: On the Web, the safe area frame value doesn't update on resize
// So we workaround this by measuring the frame on resize
function useResizeListenerWeb(onChange) {
  const frameRef = React.useRef({
    width,
    height
  });
  const elementRef = React.useRef(null);
  React.useEffect(() => {
    if (elementRef.current == null) {
      return;
    }
    const update = size => {
      if (frameRef.current.width === size.width && frameRef.current.height === size.height) {
        return;
      }
      frameRef.current = size;
      onChange(size);
    };
    const rect = elementRef.current.getBoundingClientRect();
    update({
      width: rect.width,
      height: rect.height
    });
    const observer = new ResizeObserver(entries => {
      const entry = entries[0];
      if (entry) {
        const {
          width,
          height
        } = entry.contentRect;
        update({
          width,
          height
        });
      }
    });
    observer.observe(elementRef.current);
    return () => {
      observer.disconnect();
    };
  }, [onChange]);
  const element = /*#__PURE__*/_jsx("div", {
    ref: elementRef,
    style: {
      ...StyleSheet.absoluteFillObject,
      pointerEvents: 'none',
      visibility: 'hidden'
    }
  });
  return {
    element,
    get: () => frameRef.current
  };
}
//# sourceMappingURL=useFrameSize.js.map